/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.view.aggregator;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.helix.HelixDataAccessor;
import org.apache.helix.HelixException;
import org.apache.helix.HelixManager;
import org.apache.helix.HelixManagerFactory;
import org.apache.helix.InstanceType;
import org.apache.helix.NotificationContext;
import org.apache.helix.PropertyType;
import org.apache.helix.api.config.ViewClusterSourceConfig;
import org.apache.helix.api.listeners.ClusterConfigChangeListener;
import org.apache.helix.api.listeners.PreFetch;
import org.apache.helix.common.DedupEventProcessor;
import org.apache.helix.model.ClusterConfig;
import org.apache.helix.view.aggregator.SourceClusterConfigChangeAction;
import org.apache.helix.view.aggregator.ViewClusterRefresher;
import org.apache.helix.view.common.ClusterViewEvent;
import org.apache.helix.view.dataprovider.SourceClusterDataProvider;
import org.apache.helix.view.monitoring.ViewAggregatorMonitor;
import org.apache.helix.zookeeper.zkclient.exception.ZkInterruptedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HelixViewAggregator
implements ClusterConfigChangeListener {
    private static final Logger logger = LoggerFactory.getLogger(HelixViewAggregator.class);
    private static final long DEFAULT_INITIAL_EVENT_PROCESS_BACKOFF = 10L;
    private static final long DEFAULT_MAX_EVENT_PROCESS_BACKOFF = 5000L;
    private final String _viewClusterName;
    private final HelixManager _viewClusterManager;
    private final Map<String, SourceClusterDataProvider> _dataProviderMap;
    private final DedupEventProcessor<ClusterViewEvent.Type, ClusterViewEvent> _aggregator;
    private final AtomicBoolean _refreshViewCluster;
    private final DedupEventProcessor<ClusterViewEvent.Type, ClusterViewEvent> _viewConfigProcessor;
    private final ViewAggregatorMonitor _monitor;
    private ClusterConfig _curViewClusterConfig;
    private Timer _viewClusterRefreshTimer;
    private ViewClusterRefresher _viewClusterRefresher;
    private HelixDataAccessor _dataAccessor;

    public HelixViewAggregator(String viewClusterName, String zkAddr) {
        this._viewClusterName = viewClusterName;
        this._dataProviderMap = new ConcurrentHashMap<String, SourceClusterDataProvider>();
        this._viewClusterManager = HelixManagerFactory.getZKHelixManager((String)this._viewClusterName, (String)HelixViewAggregator.generateHelixManagerInstanceName(this._viewClusterName), (InstanceType)InstanceType.SPECTATOR, (String)zkAddr);
        this._refreshViewCluster = new AtomicBoolean(false);
        this._monitor = new ViewAggregatorMonitor(viewClusterName);
        this._aggregator = new DedupEventProcessor<ClusterViewEvent.Type, ClusterViewEvent>(this._viewClusterName, "Aggregator"){

            public void handleEvent(ClusterViewEvent event) {
                HelixViewAggregator.this.handleSourceClusterEvent(event);
                HelixViewAggregator.this._monitor.recordProcessedSourceEvent();
            }
        };
        this._viewConfigProcessor = new DedupEventProcessor<ClusterViewEvent.Type, ClusterViewEvent>(this._viewClusterName, "ViewConfigProcessor"){

            public void handleEvent(ClusterViewEvent event) {
                HelixViewAggregator.this.handleViewClusterConfigChange(event);
            }
        };
    }

    public String getAggregatorInstanceName() {
        return String.format("%s::%s", this._viewClusterManager.getInstanceName(), this.hashCode());
    }

    public void start() throws Exception {
        this._monitor.register();
        this._aggregator.start();
        this._viewConfigProcessor.start();
        try {
            this._viewClusterManager.connect();
            this._dataAccessor = this._viewClusterManager.getHelixDataAccessor();
            this._viewClusterManager.addClusterfigChangeListener((ClusterConfigChangeListener)this);
        }
        catch (Exception e) {
            this.shutdown();
            throw new HelixException("Failed to connect view cluster helix manager", (Throwable)e);
        }
        this._viewClusterRefresher = new ViewClusterRefresher(this._viewClusterName, this._viewClusterManager.getHelixDataAccessor());
    }

    public void shutdown() {
        boolean success = true;
        this._aggregator.interrupt();
        this._viewConfigProcessor.interrupt();
        if (this._viewClusterRefreshTimer != null) {
            logger.info("Shutting down view cluster refresh timer");
            this._viewClusterRefreshTimer.cancel();
        }
        if (this._viewClusterManager != null && this._viewClusterManager.isConnected()) {
            logger.info("Shutting down view cluster helix manager");
            try {
                this._viewClusterManager.disconnect();
            }
            catch (ZkInterruptedException zkintr) {
                logger.warn("ZK interrupted when disconnecting helix manager", (Throwable)zkintr);
            }
            catch (Exception e) {
                success = false;
                logger.error(String.format("Failed to disconnect helix manager for view cluster %s", this._viewClusterName), (Throwable)e);
            }
        }
        for (SourceClusterDataProvider provider : this._dataProviderMap.values()) {
            logger.info(String.format("Shutting down data provider for source cluster %s", provider.getName()));
            try {
                provider.shutdown();
            }
            catch (Exception e) {
                success = false;
                logger.error(String.format("Failed to shutdown data provider %s for view cluster %s", provider.getName(), this._viewClusterName), (Throwable)e);
            }
        }
        logger.info("Unregistering monitor.");
        this._monitor.unregister();
        logger.info("HelixViewAggregator shutdown " + (success ? "cleanly" : "with error"));
    }

    @PreFetch(enabled=false)
    public void onClusterConfigChange(ClusterConfig clusterConfig, NotificationContext context) {
        if (context != null && context.getType() != NotificationContext.Type.FINALIZE) {
            this._viewConfigProcessor.queueEvent((Object)ClusterViewEvent.Type.ConfigChange, (Object)new ClusterViewEvent(this._viewClusterName, ClusterViewEvent.Type.ConfigChange));
        } else {
            logger.info(String.format("Skip processing view cluster config change with notification context type %s", context == null ? "NoContext" : context.getType().name()));
        }
    }

    private void handleSourceClusterEvent(ClusterViewEvent event) {
        logger.info(String.format("Processing event %s from source cluster %s.", event.getEventType().name(), event.getClusterName()));
        switch (event.getEventType()) {
            case ExternalViewChange: 
            case InstanceConfigChange: 
            case LiveInstanceChange: {
                this._refreshViewCluster.set(true);
                break;
            }
            case PeriodicViewRefresh: {
                if (!this._refreshViewCluster.get()) {
                    logger.info("Skip refresh: No event happened since last refresh, and no force refresh.");
                    return;
                }
                logger.info("Refreshing cluster based on event " + event.getEventType().name());
                this.refreshViewCluster();
                break;
            }
            default: {
                logger.error(String.format("Unrecognized event type: %s", new Object[]{event.getEventType()}));
            }
        }
    }

    private void handleViewClusterConfigChange(ClusterViewEvent event) {
        logger.info(String.format("Processing event %s for view cluster %s", event.getEventType().name(), this._viewClusterName));
        if (event.getEventType() == ClusterViewEvent.Type.ConfigChange) {
            boolean success;
            try {
                Thread.sleep(event.getEventProcessBackoff());
            }
            catch (InterruptedException e) {
                logger.warn("Interrupted when backing off during process view config change retry", (Throwable)e);
                Thread.currentThread().interrupt();
            }
            ClusterConfig newClusterConfig = (ClusterConfig)this._dataAccessor.getProperty(this._dataAccessor.keyBuilder().clusterConfig());
            if (newClusterConfig == null) {
                logger.warn("Failed to read view cluster config");
                success = false;
            } else {
                SourceClusterConfigChangeAction action = new SourceClusterConfigChangeAction(this._curViewClusterConfig, newClusterConfig);
                action.computeAction();
                success = this.processViewClusterConfigUpdate(action);
            }
            if (!success) {
                this._monitor.recordViewConfigProcessFailure();
                long backoff = this.computeNextEventProcessBackoff(event.getEventProcessBackoff());
                logger.info("Failed to process view cluster config change. Will retry in {} ms", (Object)backoff);
                event.setEventProcessBackoff(backoff);
                this._viewConfigProcessor.queueEvent((Object)event.getEventType(), (Object)event);
            } else {
                this._curViewClusterConfig = newClusterConfig;
            }
        } else {
            logger.error(String.format("Unrecognized event type: %s", new Object[]{event.getEventType()}));
        }
    }

    private long computeNextEventProcessBackoff(long currentBackoff) {
        if (currentBackoff <= 0L) {
            return 10L;
        }
        return currentBackoff * 2L > 5000L ? 5000L : currentBackoff * 2L;
    }

    private void resetTimer(long triggerIntervalMs) {
        if (this._viewClusterRefreshTimer != null) {
            this._viewClusterRefreshTimer.cancel();
        }
        RefreshViewClusterTask refreshTrigger = new RefreshViewClusterTask();
        this._viewClusterRefreshTimer = new Timer(true);
        this._viewClusterRefreshTimer.scheduleAtFixedRate((TimerTask)refreshTrigger, 0L, triggerIntervalMs);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean processViewClusterConfigUpdate(SourceClusterConfigChangeAction action) {
        String key;
        boolean success = true;
        for (ViewClusterSourceConfig source : action.getConfigsToDelete()) {
            key = HelixViewAggregator.generateDataProviderMapKey(source);
            logger.info("Deleting data provider " + key);
            if (!this._dataProviderMap.containsKey(key)) continue;
            try {
                this._dataProviderMap.get(key).shutdown();
                Map<String, SourceClusterDataProvider> map = this._dataProviderMap;
                synchronized (map) {
                    this._dataProviderMap.remove(key);
                    this._refreshViewCluster.set(true);
                }
            }
            catch (Exception e) {
                success = false;
                logger.warn(String.format("Failed to shutdown data provider %s, will retry", key));
            }
        }
        for (ViewClusterSourceConfig source : action.getConfigsToAdd()) {
            key = HelixViewAggregator.generateDataProviderMapKey(source);
            logger.info("Creating data provider " + key);
            if (this._dataProviderMap.containsKey(key)) {
                logger.warn(String.format("Add data provider %s which already exists. Recreating", key));
                this._dataProviderMap.remove(key);
            }
            try {
                SourceClusterDataProvider provider = new SourceClusterDataProvider(source, this._aggregator);
                provider.setup();
                this._dataProviderMap.put(key, provider);
            }
            catch (Exception e) {
                success = false;
                logger.warn(String.format("Failed to create data provider %s, will retry", key));
            }
        }
        if (action.shouldResetTimer()) {
            logger.info("Resetting view cluster refresh timer at interval " + action.getCurrentRefreshPeriodMs());
            this.resetTimer(action.getCurrentRefreshPeriodMs());
        }
        return success;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void refreshViewCluster() {
        HashSet<SourceClusterDataProvider> providerView;
        long startRefreshMs = System.currentTimeMillis();
        logger.info(String.format("START RefreshViewCluster: Refresh view cluster %s at timestamp %s", this._viewClusterName, startRefreshMs));
        boolean dataProviderFailure = false;
        boolean viewClusterFailure = false;
        Iterator iterator = this._dataProviderMap;
        synchronized (iterator) {
            this._refreshViewCluster.set(false);
            providerView = new HashSet<SourceClusterDataProvider>(this._dataProviderMap.values());
        }
        for (SourceClusterDataProvider provider : providerView) {
            try {
                provider.refreshCache();
            }
            catch (Exception e) {
                logger.warn("Caught exception when refreshing source cluster cache. Abort refresh.", (Throwable)e);
                this._refreshViewCluster.set(true);
                dataProviderFailure = true;
                break;
            }
        }
        if (!dataProviderFailure) {
            this._viewClusterRefresher.updateProviderView(providerView);
            for (PropertyType propertyType : ViewClusterSourceConfig.getValidPropertyTypes()) {
                logger.info(String.format("Refreshing property %s in view cluster %s", propertyType, this._viewClusterName));
                try {
                    if (this._viewClusterRefresher.refreshPropertiesInViewCluster(propertyType)) continue;
                    viewClusterFailure = true;
                    this._refreshViewCluster.set(true);
                }
                catch (IllegalArgumentException e) {
                    logger.error(String.format("Failed to refresh property in view cluster %s with exception", this._viewClusterName), (Throwable)e);
                }
            }
        }
        this.recordRefreshResults(dataProviderFailure, viewClusterFailure, System.currentTimeMillis() - startRefreshMs);
    }

    private void recordRefreshResults(boolean recordSourceFailure, boolean recordViewFailure, long latency) {
        if (recordSourceFailure) {
            this._monitor.recordReadSourceFailure();
        }
        if (recordViewFailure) {
            this._monitor.recordViewRefreshFailure();
        }
        this._monitor.recordRefreshViewLatency(latency);
        logger.info(String.format("END RefreshViewCluster: finished refresh %s. Time spent: %s ms. Success: %s", this._viewClusterName, latency, !recordSourceFailure && !recordViewFailure));
    }

    private static String generateHelixManagerInstanceName(String viewClusterName) {
        return String.format("HelixViewAggregator-%s", viewClusterName);
    }

    private static String generateDataProviderMapKey(ViewClusterSourceConfig config) {
        return String.format("%s-%s", config.getName(), config.getZkAddress());
    }

    private class RefreshViewClusterTask
    extends TimerTask {
        private RefreshViewClusterTask() {
        }

        @Override
        public void run() {
            logger.info("Triggering view cluster refresh");
            HelixViewAggregator.this._aggregator.queueEvent((Object)ClusterViewEvent.Type.PeriodicViewRefresh, (Object)new ClusterViewEvent(HelixViewAggregator.this._viewClusterName, ClusterViewEvent.Type.PeriodicViewRefresh));
        }
    }
}

